<?php

namespace App\Http\Actions\Admin;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Order;
use App\Models\Category;
use App\Models\Shop;
use Illuminate\View\View;

class AdminDashboardAction
{
    /**
     * Handle the incoming dashboard request.
     */
    public function __invoke(): View
    {
        $year        = Carbon::now()->year;
        $months      = $this->farsiMonths();

        // سفارش‌ها
        $newOrdersCount       = $this->countTodayAwaiting();
        $todayOrdersCount     = Order::today()->count();
        $thisMonthOrdersCount = Order::thisMonth()->count();
        $totalOrdersCount     = Order::count();
        $abandonedOrdersCount = Order::awaitingPayment()->count();

        // فروش‌ها
        $todaySales   = Order::today()->awaitingFulfillment()->sum('amount');
        $monthSales   = Order::thisMonth()->awaitingFulfillment()->sum('amount');
        $totalSales   = Order::awaitingFulfillment()->sum('amount');

        $monthlySales = $this->getYearlySalesByMonth($year);

        $categories = Category::withCount('products')->get()->pluck('products_count', 'name');


        $newCustomers = User::latest()->take(10)->get();

        $newShops = Shop::latest()->take(10)->get();
        $approvedShopsCount = Shop::approved()->count();
        $notApprovedShopsCount = Shop::notApproved()->count();

        return view('pages.admin.dashboard', compact(
            'newOrdersCount',
            'todayOrdersCount',
            'thisMonthOrdersCount',
            'totalOrdersCount',
            'abandonedOrdersCount',
            'todaySales',
            'monthSales',
            'totalSales',
            'monthlySales',
            'months',
            'categories',
            'newCustomers',
            'newShops',
            'approvedShopsCount',
            'notApprovedShopsCount',
        ));
    }

    /**
     * Count orders created today with "AwaitingPayment" status.
     */
    private function countTodayAwaiting(): int
    {
        return Order::today()
            ->awaitingPayment()
            ->count();
    }

    /**
     * Get an array of monthly sales totals for the given year.
     *
     * @return int[] [12 values indexed 0–11]
     */
    private function getYearlySalesByMonth(int $year): array
    {
        $salesByMonth = Order::selectRaw('MONTH(created_at) as month, SUM(amount) as total')
            ->whereYear('created_at', $year)
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month');

        // ensure all 12 months are present
        return collect(range(1, 12))
            ->map(fn($m) => (int) ($salesByMonth->get($m) ?: 0))
            ->all();
    }

    /**
     * Persian month names.
     *
     * @return string[]
     */
    private function farsiMonths(): array
    {
        return [
            'فروردین',
            'اردیبهشت',
            'خرداد',
            'تیر',
            'مرداد',
            'شهریور',
            'مهر',
            'آبان',
            'آذر',
            'دی',
            'بهمن',
            'اسفند',
        ];
    }
}
