<?php

namespace App\Http\Controllers\Admin\WooCommerce;

use App\Http\Controllers\Controller;
use App\Services\Woo\WooClient;
use App\Services\Woo\WooCredentials;
use Illuminate\Contracts\Cache\Repository as CacheRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Crypt;

abstract class WooBaseController extends Controller
{
    protected $shop;
    protected WooCredentials $wooCredentials;
    protected WooClient $wooClient;

    public function __construct()
    {
        $this->shop = auth()->user()->shop;

        $this->wooCredentials = new WooCredentials(
            baseUrl: $this->shop->woo_url,
            consumerKey: decrypt($this->shop->consumer_key),
            consumerSecret: decrypt($this->shop->consumer_secret)
        );

        $this->wooClient = new WooClient($this->wooCredentials);
    }

    /** Per-shop cache tags (works if driver supports tags). */
    protected function cacheTags(): array
    {
        return ['woo', 'shop:' . $this->shop->id];
    }

    /** Build a consistent cache key. */
    protected function cacheKey(string $area, array $parts = []): string
    {
        $base = [
            'woo',
            $area,
            'shop:' . $this->shop->id,
            'url:' . md5($this->shop->woo_url),
        ];
        foreach ($parts as $k => $v) {
            $base[] = is_string($k) ? "{$k}:" . (string)($v ?? 'null') : (string)$v;
        }
        return implode('|', $base);
    }

    /**
     * Remember with tags when available, falling back to plain remember.
     * $ttlSeconds can be int or Carbon|\DateInterval.
     */
    protected function remember(string $key, $ttlSeconds, \Closure $callback)
    {
        $store = Cache::getStore();
        if (method_exists($store, 'tags')) {
            return Cache::tags($this->cacheTags())->remember(
                $key,
                $ttlSeconds instanceof \DateInterval || $ttlSeconds instanceof Carbon
                    ? $ttlSeconds
                    : Carbon::now()->addSeconds((int)$ttlSeconds),
                $callback
            );
        }
        return Cache::remember($key, (int)$ttlSeconds, $callback);
    }

    /** Forget a specific key (tagged if possible). */
    protected function forget(string $key): void
    {
        $store = Cache::getStore();
        if (method_exists($store, 'tags')) {
            Cache::tags($this->cacheTags())->forget($key);
        } else {
            Cache::forget($key);
        }
    }

    /** Flush all Woo caches for this shop (use carefully). */
    protected function flushShopCache(): void
    {
        $store = Cache::getStore();
        if (method_exists($store, 'tags')) {
            Cache::tags($this->cacheTags())->flush();
        }
    }

    /** Helper to read default TTL (seconds). */
    protected function defaultTtl(): int
    {
        return (int) config('services.woo.cache_ttl', 300);
    }
}
