<?php

namespace App\Livewire\Blesser;

use App\Models\Shop;
use App\Models\Product;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Events\UserVisitedProduct;
use App\Services\Models\CartService;
use App\Models\ProductInventoryVariant;
use App\Services\Models\ProductService;
use App\Services\Models\CategoryAttributeService;

class ProductDetails extends Component
{
    protected CartService $cart;
    public bool $eventDispatched = false;

    public $product;
    public $inventory;
    public $inventoryVariants;
    public $gallery;

    public $sellers;

    // this the product variants it hold data to generate the variant selection component in blade
    public $productVariants;

    public $selectedSeller = null;

    public $price;
    public $discountAmount = 0;

    // this stores the variant selection its an array attribute_id => attribute_item_id
    public $selectedVariants;

    // this stores the variant selection combinationKey
    public $combinationKey;

    // when a user changes product variants if there is variant with the selected combination key the ids of variants will be stored here
    public $availableVariants;

    public bool $outOfStock = true;
    public function __construct()
    {
        $this->cart = app(CartService::class);
    }

    public function mount($slug)
    {
        $this->prepareData($slug);
        $this->updateSEO();
        $this->updateSellers();
        $this->getProductVariants();

        if (!$this->eventDispatched) {
            event(new UserVisitedProduct($this->product));
            $this->eventDispatched = true;
        }
    }

    public function selectSeller($sellerId)
    {
        $this->selectedSeller = $sellerId;
        $this->updatePrice();
    }

    #[On('seller-selected')]
    public function updatePrice()
    {
        if ($this->selectedSeller == null) {
            return;
        }
        $inventory = $this->inventory->get($this->selectedSeller);

        $this->inventoryVariants->each(function ($variant) use ($inventory, &$test) {
            if ($variant->combination_key == $this->combinationKey && $variant->product_inventory_id == $inventory->id) {
                $this->price = $variant->price_override ?? $this->product->price;
                $this->discountAmount = $variant->discount_amount ?? 0;
            }
        });
    }

    public function updatedSelectedVariants()
    {
        $selectedVariants = $this->selectedVariants;

        ksort($selectedVariants); // stable JSON order

        $this->combinationKey = collect($selectedVariants)->values()->sort()->implode('-');


        if (!isset($this->inventoryVariants->groupBy('combination_key')[$this->combinationKey])) {
            $this->outOfStock = true;
            $this->availableVariants = [];
            return;
        } else {
            $this->outOfStock = false;
        }

        $this->availableVariants = $this->inventoryVariants->groupBy('combination_key')[$this->combinationKey]->pluck('id') ?? [];

        $this->updateSellers();
        $this->updatePrice();
    }

    public function addToCart()
    {
        $this->cart->add(
            productId: $this->product->id,
            inventoryId: $this->inventory[$this->selectedSeller]->id ?? null,
            shopId: $this->selectedSeller,
            variantId: $this->inventoryVariants->where('combination_key', $this->combinationKey)->where('product_inventory_id', $this->inventory[$this->selectedSeller]->id ?? null)->first()->id ?? null,
            quantity: 1,
            variant: [],
            price: $this->price,
            discount: $this->discountAmount,
            coupon: null,
            options: []
        );

        return redirect()->route('cart.index')->with('success', __('Added to cart.'));
    }
    public function render()
    {

        return view('livewire.blesser.product-details')->extends('layouts.blesser.app')->section('content');
    }



    // --------------------
    private function getInventoryVariant()
    {
        $query = ProductInventoryVariant::query();

        $query->when(value: $this->selectedSeller, callback: function ($query, $seller) {
            $query->whereHas(relation: 'inventory', callback: function ($query) use ($seller) {
                $query->where(column: 'shop_id', operator: $seller);
            });
        });

        $inventoryVariants = $query->get();

        $this->inventoryVariants = $inventoryVariants;
    }
    private function getProductVariants()
    {
        $this->productVariants = CategoryAttributeService::getCategoryPriceEffectiveAttributes($this->product->category)->map(function ($attribute) {
            return [
                'id' => $attribute->id,
                'name' => $attribute->name,
                'label' => $attribute->name,
                'items' => $attribute->items->map(function ($item) {
                    return [
                        'id' => $item->id,
                        'group' => $item->item_group,
                        'key' => $item->item_key,
                        'value' => $item->item_value,
                    ];
                })->keyBy('id')->toArray(),
            ];
        })->keyBy('name')->toArray();
    }
    private function updateSellers()
    {
        $this->sellers = $this->product->sellers()->with('inventoryVariants')->whereHas('inventoryVariants')
            ->when(value: $this->availableVariants, callback: function ($query) {
                $query->whereHas(relation: 'inventoryVariants', callback: function ($query) {
                    $query->whereIn(column: 'product_inventory_variants.id', values: $this->availableVariants);
                });
            })->get()->keyBy(keyBy: 'id');
    }
    private function prepareData($slug)
    {
        $this->product = Product::where('slug', $slug)->firstOrFail();
        $this->inventory = $this->product->inventories->keyBy('shop_id');
        $this->gallery = $this->product->getMetaByKey('gallery');

        $this->getInventoryVariant();


        $firstVariant = $this->product->inventoryVariants->first();

        if (!$firstVariant) {
            $this->outOfStock = true;
            return;
        }

        $this->price = $firstVariant->price_override ?? $this->product->price;
        $this->selectedVariants = $firstVariant->combination_items;
        $this->combinationKey = $firstVariant->combination_key;

        $this->outOfStock = false;

        $this->selectedSeller = $this->inventory->where('id', $firstVariant->product_inventory_id)->first()->shop_id ?? null;

        $this->updatedSelectedVariants();
    }

    private function updateSEO()
    {
        $meta = [
            // 👇 use the array above
            'type'        => 'product',
            'title'       => "{$this->product->title} | " . getSiteName(),
            'description' => $this->product->short_description ?? Str::limit(strip_tags($this->product->short_description), 160) ?? '',
            'keywords'    => implode(', ', [
                $this->product->category->name ?? '',
                $this->product->title,
            ]),
            'image' => $this->product->image_url ?? asset('images/og-default.jpg'),

            'og:type' => "product",
            'og:title' => "{$this->product->title} | " . getSiteName(),
            'og:description' => $this->product->short_description ?? Str::limit(strip_tags($this->product->short_description), 160) ?? '',
            'og:image' => $this->product->image_url ?? asset('images/og-default.jpg'),

            'twitter:title' => "{$this->product->title} | " . getSiteName(),
            'twitter:description' => $this->product->short_description ?? Str::limit(strip_tags($this->product->short_description), 160) ?? '',
            'twitter:image' => $this->product->image_url ?? asset('images/og-default.jpg'),
        ];

        seo()->setTitle(title: $this->product->title)->meta()->setMany($meta);
    }
}
