<?php

declare(strict_types=1);

namespace App\Services\Woo;

use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Http\Client\RequestException;
use Illuminate\Http\Client\ConnectionException;
use App\Exceptions\Woocommerce\WooException;
use App\Exceptions\Woocommerce\WooAuthException;
use App\Exceptions\Woocommerce\WooServerException;
use App\Exceptions\Woocommerce\WooNotFoundException;
use App\Exceptions\Woocommerce\WooRateLimitException;

final class WooClient
{
    public function __construct(
        private WooCredentials $creds,
        private int $connectTimeout = 5,
        private int $readTimeout = 25,
        private int $maxRetries = 3
    ) {}

    /** Core GET with retries/backoff and error mapping */
    public function get(string $endpoint, array $query = []): array
    {
        $url = rtrim($this->creds->baseUrl, '/') . '/wp-json/wc/v3/' . ltrim($endpoint, '/');

        $request = $this->http()->withHeaders([
            'User-Agent'   => 'YourMarketplace/1.0',
            'X-Request-ID' => request()?->header('X-Request-ID') ?? bin2hex(random_bytes(8)),
        ]);

        // Authentication
        if ($this->creds->isHttps()) {
            $request = $request->withBasicAuth($this->creds->consumerKey, $this->creds->consumerSecret);
        } elseif ($this->creds->allowQueryAuthFallback) {
            $query['consumer_key']    = $this->creds->consumerKey;
            $query['consumer_secret'] = $this->creds->consumerSecret;
        } else {
            throw new WooAuthException('Non-HTTPS without query auth fallback is not allowed.');
        }

        // Retry with backoff and throw
        $response = $request
            ->retry(
                times: $this->maxRetries,
                sleepMilliseconds: 300,
                when: function ($exception, PendingRequest $req) {
                    if ($exception instanceof ConnectionException) return true;
                    if ($exception instanceof RequestException) {
                        $status = optional($exception->response)->status();
                        return $status === 429 || ($status >= 500 && $status < 600);
                    }
                    return false;
                },
                throw: true
            )
            ->get($url, $query)
            ->throw();

        $this->throwIfError($response);

        /** @var array $data */
        $data = $response->json();
        return $data ?? [];
    }

    public function getRaw(string $endpoint, array $query = []): \Illuminate\Http\Client\Response
    {
        // same auth & retry as get(), but return the Response
        $url = rtrim($this->creds->baseUrl, '/') . '/wp-json/wc/v3/' . ltrim($endpoint, '/');

        $request = $this->http()->withHeaders([
            'User-Agent'   => 'YourMarketplace/1.0',
            'X-Request-ID' => request()?->header('X-Request-ID') ?? bin2hex(random_bytes(8)),
        ]);

        if ($this->creds->isHttps()) {
            $request = $request->withBasicAuth($this->creds->consumerKey, $this->creds->consumerSecret);
        } elseif ($this->creds->allowQueryAuthFallback) {
            $query['consumer_key']    = $this->creds->consumerKey;
            $query['consumer_secret'] = $this->creds->consumerSecret;
        } else {
            throw new \App\Exceptions\Woocommerce\WooAuthException('Non-HTTPS without query auth fallback is not allowed.');
        }

        $resp = $request
            ->retry(
                times: $this->maxRetries,
                sleepMilliseconds: 300,
                when: function ($exception, \Illuminate\Http\Client\PendingRequest $req) {
                    return $exception instanceof \Illuminate\Http\Client\ConnectionException
                        || ($exception instanceof \Illuminate\Http\Client\RequestException
                            && in_array(optional($exception->response)->status(), [429, 500, 501, 502, 503, 504, 508, 509], true));
                },
                throw: true
            )
            ->get($url, $query)
            ->throw();

        $this->throwIfError($resp);
        return $resp;
    }

    private function http(): PendingRequest
    {
        return Http::acceptJson()
            ->connectTimeout($this->connectTimeout)
            ->timeout($this->readTimeout);
    }

    private function throwIfError(Response $resp): void
    {
        if ($resp->successful()) return;

        $msg = $resp->body();
        $status = $resp->status();

        if ($status === 401 || $status === 403) {
            throw new WooAuthException($msg, $status);
        }
        if ($status === 404) {
            throw new WooNotFoundException($msg, 404);
        }
        if ($status === 429) {
            throw new WooRateLimitException($msg, 429);
        }
        if ($resp->serverError()) {
            throw new WooServerException($msg, $status);
        }
        throw new WooException($msg, $status);
    }

    /* --------- High-level helpers --------- */

    /** Auto-paginated iterator */
    public function paged(string $endpoint, array $query = [], int $perPage = 100): \Generator
    {
        $page = 1;
        do {
            $resp = $this->get($endpoint, array_merge($query, ['per_page' => $perPage, 'page' => $page]));
            $count = is_array($resp) ? count($resp) : 0;
            if ($count === 0) break;
            foreach ($resp as $item) yield $item;
            $page++;
        } while ($count === $perPage);
    }

    /** Incremental products since timestamp (UTC ISO8601) */
    public function productsSince(?string $modifiedAfterIso = null): \Generator
    {
        $q = [];
        if ($modifiedAfterIso) $q['modified_after'] = $modifiedAfterIso;
        return $this->paged('products', $q);
    }
    public function ordersSince(?string $modifiedAfterIso = null): \Generator
    {
        $q = [];
        if ($modifiedAfterIso) $q['modified_after'] = $modifiedAfterIso;
        return $this->paged('orders', $q);
    }
    public function customersSince(?string $modifiedAfterIso = null): \Generator
    {
        $q = [];
        if ($modifiedAfterIso) $q['modified_after'] = $modifiedAfterIso;
        return $this->paged('customers', $q);
    }

    public function productsPage(int $page = 1, int $perPage = 20, array $query = []): array
    {
        $resp = $this->getRaw('products', array_merge($query, [
            'page'     => $page,
            'per_page' => $perPage,
        ]));

        return [
            'data'        => $resp->json() ?? [],
            'total'       => (int) $resp->header('X-WP-Total', 0),
            'total_pages' => (int) $resp->header('X-WP-TotalPages', 0),
        ];
    }
    public function systemStatus(): array
    {
        return $this->get('system_status');
    }

    public function product(int $productId): array
    {
        return $this->get("products/{$productId}");
    }

    public function productVariations(int $productId): array
    {
        return iterator_to_array($this->paged("products/{$productId}/variations"));
    }
}
