<?php
namespace App\Traits;

use App\Models\Tag;

trait Taggable
{
    /**
     * Get all of the tags for the model.
     */
    public function tags()
    {
        return $this->morphToMany(Tag::class, 'taggable');
    }

    /**
     * Attach tags to the model.
     *
     * @param array|string $tags
     * @return void
     */
    public function attachTags($tags)
    {
        $this->tags()->sync($tags);
    }

    /**
     * Detach tags from the model.
     *
     * @param array|string $tags
     * @return void
     */
    public function detachTags($tags = null)
    {
        if (is_null($tags)) {
            $this->tags()->detach();
            return;
        }

        $this->tags()->detach($tags);
    }

    /**
     * Sync tags with the model.
     *
     * @param array|string $tags
     * @return void
     */
    public function syncTags($tags)
    {
        $this->tags()->sync($tags);
    }

    /**
     * Check if model has any tags.
     *
     * @return bool
     */
    public function hasTags()
    {
        return $this->tags()->exists();
    }

    /**
     * Check if model has a specific tag.
     *
     * @param string|int $tag
     * @return bool
     */
    public function hasTag($tag)
    {
        if (is_numeric($tag)) {
            return $this->tags()->where('id', $tag)->exists();
        }

        return $this->tags()->where('name', $tag)->exists();
    }
}
