/**
 * @typedef {Object} AttributeOption
 * @property {'text'|'number'} type
 * @property {string|number} [default]
 */

/**
 * @typedef {Object} LaravelBlockOptions
 * @property {string} name - Unique block identifier
 * @property {string} label - Human-readable label
 * @property {string} tag - Blade component tag name
 * @property {string} [category] - Block category
 * @property {Object.<string, AttributeOption>} [attrs] - HTML attributes
 * @property {Object.<string, AttributeOption>} [configProps] - Configuration properties stored in data-config
 * @property {string} [icon] - Block icon
 */

/**
 * Safely stringify JSON for HTML attribute display
 * @param {string} str - JSON string to format
 * @returns {string} Formatted string
 */
export function safeStringifyJSONAttribute(str) {
    try {
        const parsed = JSON.parse(str);
        const stringified = JSON.stringify(parsed);
        return stringified.length > 50 ? stringified.substring(0, 47) + '...' : stringified;
    } catch {
        return '{}';
    }
}

/**
 * Get parsed config from component
 * @param {Component} component - GrapesJS component
 * @returns {Object} Parsed config object
 */
export function getComponentConfigJSON(component) {
    try {
        return JSON.parse(component.getAttributes()['data-config'] || '{}');
    } catch {
        return {};
    }
}

/**
 * Set component config
 * @param {Component} component - GrapesJS component
 * @param {Object} config - Config object to set
 */
export function setComponentConfigJSON(component, config) {
    component.addAttributes({ 'data-config': JSON.stringify(config) });
}

/**
 * Create a Laravel Blade component block factory
 * @param {Editor} editor - GrapesJS editor instance
 * @param {LaravelBlockOptions} options - Block configuration options
 */
export function makeLaravelBlock(editor, options) {
    const {
        name,
        label,
        tag,
        category = 'Basic',
        attrs = {},
        configProps = {},
        icon
    } = options;

    // Register custom trait type for config properties
    if (!editor.Traits.getType('cfg')) {
        editor.Traits.addType('cfg', {
            createInput({ trait }) {
                const traitConfig = trait.get('props') || {};
                const input = document.createElement('input');
                input.type = traitConfig.inputType || 'text';
                input.placeholder = traitConfig.placeholder || '';

                // Add data attribute for easier selection
                input.setAttribute('data-cfg-key', traitConfig.cfgKey);

                return input;
            },

            onEvent({ component, elInput, trait }) {
                const traitConfig = trait.get('props') || {};
                const cfgKey = traitConfig.cfgKey;

                if (!cfgKey) return;

                // Get current config
                const currentConfig = getComponentConfigJSON(component);

                // Update value based on input type
                let newValue = elInput.value;
                if (traitConfig.isNumber) {
                    newValue = newValue ? parseFloat(newValue) : traitConfig.default || 0;
                }

                // Update config and set attribute
                const updatedConfig = { ...currentConfig, [cfgKey]: newValue };
                setComponentConfigJSON(component, updatedConfig);

                // Trigger update for the view
                component.trigger('change:attributes');
            },

            onUpdate({ component, elInput, trait }) {
                const traitConfig = trait.get('props') || {};
                const cfgKey = traitConfig.cfgKey;

                if (!cfgKey || !elInput) return;

                const currentConfig = getComponentConfigJSON(component);
                const value = currentConfig[cfgKey];

                if (value !== undefined && value !== null) {
                    elInput.value = value.toString();
                } else {
                    elInput.value = traitConfig.default || '';
                }
            }
        });
    }

    // Build traits array for this specific block
    const buildTraits = () => {
        const traits = [];

        // Add attributes traits
        Object.entries(attrs).forEach(([attrName, attrConfig]) => {
            traits.push({
                type: attrConfig.type === 'number' ? 'number' : 'text',
                name: attrName,
                label: attrName.replace('data-', '').replace(/-/g, ' '),
                ...(attrConfig.default !== undefined && { default: attrConfig.default })
            });
        });

        // Add config properties traits
        Object.entries(configProps).forEach(([propName, propConfig]) => {
            traits.push({
                type: 'cfg',
                name: `cfg:${propName}`,
                label: propName,
                props: {
                    cfgKey: propName,
                    isNumber: propConfig.type === 'number',
                    inputType: propConfig.type === 'number' ? 'number' : 'text',
                    placeholder: propConfig.default?.toString() || '',
                    default: propConfig.default
                }
            });
        });

        // Add raw JSON trait
        traits.push({
            type: 'textarea',
            name: 'data-config',
            label: 'Config (JSON)'
        });

        return traits;
    };

    // Create a unique component type for each block to preserve traits
    const componentTypeName = `laravel-blade-${name}`;

    if (!editor.DomComponents.getType(componentTypeName)) {
        editor.DomComponents.addType(componentTypeName, {
            isComponent: (el) => {
                // Match both the specific tag and generic x-component with our data attributes
                return el.tagName?.toLowerCase() === tag?.toLowerCase() ||
                    (el.getAttribute('data-laravel-block') === name);
            },

            model: {
                defaults: {
                    tagName: tag,
                    droppable: false,
                    highlightable: true,
                    void: false,
                    attributes: {
                        'data-config': '{}',
                        'data-laravel-block': name
                    },
                    traits: buildTraits(),
                    name: label
                }
            },

            view: {
                init() {
                    this.placeholderRendered = false;
                    this.listenTo(this.model, 'change:attributes', this.handleAttributeChange);
                },

                onRender() {
                    if (this.placeholderRendered) return;

                    this.createPlaceholder();
                    this.placeholderRendered = true;
                },

                createPlaceholder() {
                    // Create container
                    const container = document.createElement('div');
                    container.style.cssText = `
            min-height: 44px;
            border: 1px dashed #999;
            padding: 8px 12px;
            font-family: system-ui, sans-serif;
            font-size: 12px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            background: #f9f9f9;
          `;

                    // Left side: tag and title
                    const leftSide = document.createElement('div');
                    leftSide.style.display = 'flex';
                    leftSide.style.alignItems = 'center';
                    leftSide.style.gap = '8px';

                    this.tagElement = document.createElement('strong');
                    this.tagElement.style.color = '#333';

                    this.titleElement = document.createElement('span');
                    this.titleElement.style.color = '#666';

                    leftSide.appendChild(this.tagElement);
                    leftSide.appendChild(this.titleElement);

                    // Right side: config preview
                    const rightSide = document.createElement('div');
                    this.configElement = document.createElement('code');
                    this.configElement.style.cssText = `
            background: #fff;
            padding: 2px 6px;
            border-radius: 3px;
            border: 1px solid #ddd;
            color: #888;
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
          `;

                    rightSide.appendChild(this.configElement);
                    container.appendChild(leftSide);
                    container.appendChild(rightSide);

                    this.el.appendChild(container);
                    this.updatePlaceholderContent();
                },

                handleAttributeChange() {
                    this.updatePlaceholderContent();
                },

                updatePlaceholderContent() {
                    if (!this.placeholderRendered) return;

                    const attributes = this.model.getAttributes();
                    const dataTitle = attributes['data-title'] || '';
                    const tagName = attributes.tagName || tag;

                    // Update tag element
                    this.tagElement.textContent = tagName.toLowerCase();

                    // Update title element
                    if (dataTitle) {
                        this.titleElement.textContent = `· ${dataTitle}`;
                    } else {
                        this.titleElement.textContent = '';
                    }

                    // Update config preview
                    const configStr = attributes['data-config'] || '{}';
                    this.configElement.textContent = `data-config='${safeStringifyJSONAttribute(configStr)}'`;
                },

                remove() {
                    this.stopListening(this.model, 'change:attributes', this.handleAttributeChange);
                }
            }
        });
    }

    // Build default attributes
    const defaultAttributes = {
        'data-laravel-block': name
    };

    // Set scalar attributes
    Object.entries(attrs).forEach(([attrName, attrConfig]) => {
        defaultAttributes[attrName] = attrConfig.default || '';
    });

    // Build initial config
    const initialConfig = {};
    Object.entries(configProps).forEach(([propName, propConfig]) => {
        initialConfig[propName] = propConfig.default !== undefined ?
            propConfig.default :
            (propConfig.type === 'number' ? 0 : '');
    });

    defaultAttributes['data-config'] = JSON.stringify(initialConfig);

    // Register the block
    editor.Blocks.add(`laravel-${name}`, {
        label,
        category,
        attributes: {
            class: icon ? `fa ${icon}` : '',
            title: label
        },
        content: {
            type: componentTypeName,
            attributes: defaultAttributes
        }
    });
}
