export function makeLaravelBlock(editor, {
    name,
    label,
    tag,
    category = 'Laravel Components',
    configProps = {},
    attrs = {}
}) {
    // Build initial config
    const configDefaults = {};
    Object.entries(configProps).forEach(([k, def]) => {
        configDefaults[k] = def.default;
    });

    const attrDefaults = {};
    Object.entries(attrs).forEach(([k, def]) => {
        attrDefaults[k] = def.default;
    });

    const attributes = {
        ...attrDefaults,
        ':config': makeConfigString(configDefaults),
    };

    editor.DomComponents.addType(name, {
        isComponent: el => {
            if (el.tagName === tag.toUpperCase()) {
                return { type: name };
            }
        },
        model: {
            defaults: {
                tagName: tag,
                attributes,
                editable: false,
                droppable: false,

                // Placeholder box inside editor
                components: `
          <div class="gjs-placeholder"
            style="background:#f3f3f3;border:1px dashed #aaa;
                   padding:20px;text-align:center;color:#555;font-size:14px;">
            ${label} ${makeConfigString(configDefaults)}
          </div>
        `,

                traits: [
                    // Config traits
                    ...Object.entries(configProps).map(([key, def]) => ({
                        type: def.type || 'text',
                        label: key.charAt(0).toUpperCase() + key.slice(1),
                        name: `config-${key}`,
                        value: def.default,
                        changeProp: 1,
                    })),
                    // Attr traits
                    ...Object.entries(attrs).map(([key, def]) => ({
                        type: def.type || 'text',
                        label: key.charAt(0).toUpperCase() + key.slice(1),
                        name: key,
                        value: def.default,
                        changeProp: 1,
                    })),
                ],
            },

            init() {
                // Update config string when config traits change
                Object.keys(configProps).forEach(key => {
                    this.on(`change:attributes:config-${key}`, (model, value) => {
                        const config = {};
                        Object.keys(configProps).forEach(k => {
                            config[k] = this.getAttributes()[`config-${k}`] || configProps[k].default;
                        });
                        this.setAttributes({ ':config': makeConfigString(config) });
                        updatePlaceholder(this, label, config, this.getAttributes());
                    });
                });

                // Update simple attrs like title
                Object.keys(attrs).forEach(key => {
                    this.on(`change:attributes:${key}`, (model, value) => {
                        this.setAttributes({ [key]: value || attrs[key].default });
                        updatePlaceholder(this, label, configProps, this.getAttributes());
                    });
                });
            },
        },
    });

    // Register block
    editor.BlockManager.add(name, {
        label,
        category,
        content: {
            type: name,
            attributes,
        },
    });
}

// Convert object → PHP-style array string
function makeConfigString(config) {
    const parts = Object.entries(config).map(([k, v]) => {
        if (typeof v === 'string') return `'${k}' => '${v}'`;
        return `'${k}' => ${v}`;
    });
    return `[${parts.join(', ')}]`;
}

// Update placeholder text inside editor
function updatePlaceholder(model, label, config, attrs) {
    const el = model.view.el.querySelector('.gjs-placeholder');
    if (el) {
        el.innerText = `${label}\n${makeConfigString(config)}\n` +
            Object.entries(attrs)
                .filter(([k]) => k !== ':config')
                .map(([k, v]) => `${k}="${v}"`)
                .join(' ');
    }
}
