<?php

namespace Artemis\Faq\Traits;

use Artemis\Faq\Models\Comment;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\MorphMany;

/**
 * @mixin Model
 */
trait Commentable
{
    /**
     * Define a polymorphic one-to-many relationship for comments.
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Comment::class, 'commentable');
    }
    
    /**
     * Add a comment to the model.
     */
    public function addComment(string $content, int $creatorId): Comment
    {
        return $this->comments()->create([
            'body' => $content,
            'user_id' => $creatorId,
        ]);
    }
    
    /**
     * Update a specific comment.
     */
    public function updateComment(int $commentId, string $newContent): bool
    {
        return $this->comments()->findOrFail($commentId)->update(['comment' => $newContent]);
    }
    
    /**
     * Remove a specific comment.
     */
    public function removeComment(int $commentId): ?bool
    {
        return $this->comments()->findOrFail($commentId)->delete();
    }
    
    /**
     * Check if the model has any comments.
     */
    public function hasComments(): bool
    {
        return $this->comments()->exists();
    }
    
    /**
     * Retrieve the latest comment on the model.
     */
    public function latestComment(): ?Comment
    {
        return $this->comments()->latest()->first();
    }
    
    /**
     * Retrieve the oldest comment on the model.
     */
    public function oldestComment(): ?Comment
    {
        return $this->comments()->oldest()->first();
    }
    
    /**
     * Count the total number of comments on the model.
     */
    public function commentCount(): int
    {
        return $this->comments()->count();
    }
    
    /**
     * Retrieve comments made by a specific user.
     */
    public function commentsByUser(int $userId): Collection
    {
        return $this->comments()->where('creator', $userId)->get();
    }
    
    /**
     * Remove all comments from the model.
     */
    public function clearComments(): ?bool
    {
        return $this->comments()->delete();
    }
}